import arcpy
import os
from utils import msg

arcpy.env.overwriteOutput = True

def calc_index(b1, b2, in_path, out_dir, output_suffix):
    '''
    Calculate a normalized difference index using arcpy. Exports to a cloud-reference-format (CRF) multidimensional raster.

    b1 & b2         = name of a single band raster used for creating the index
    in_path         = path to a mosaic dataset containing the bands
    out_dir         = folder for saving the output CRFs
    output_suffix   = adds descriptor onto the output file name

    '''
    b1_raster = arcpy.Raster(in_path + b1)
    b2_raster = arcpy.Raster(in_path + b2)
    index_temp = arcpy.sa.RasterCalculator([b1_raster, b2_raster], ['x', 'y'], "(x - y) / (x + y)") 
    name = os.path.basename(os.path.normpath(in_path))
    out_path = os.path.join(out_dir, name + output_suffix)
    # Save layer to .crf
    arcpy.management.CopyRaster(index_temp, out_path, format='CRF', process_as_multidimensional="ALL_SLICES")

    return msg(f'\tCreated multidimensional raster at "{out_path}"')

def calc_md_index(mosaic_path, output_dir, ndvi_toggle, ndmi_toggle, nddsi_toggle):
    # Calculate NDVI and save to CRF multidim raster
    if ndvi_toggle:
        msg("Calculating NDVI...")
        calc_index("/Band_5", "/Band_4", mosaic_path, output_dir, "_ndvi.crf")

    # Calculate NDMI and save to CRF multidim raster
    if ndmi_toggle:
        msg('Calculating NDMI...')
        calc_index("/Band_5", "/Band_6", mosaic_path, output_dir, "_ndmi.crf")

    # Calculate NDDSI and save to CRF multidim raster
    if nddsi_toggle:
        msg("Calculating NDDSI...")
        calc_index("/Band_7", "/Band_6", mosaic_path, output_dir, "_nddsi.crf")
