import arcpy
import os
from utils import msg, warn

arcpy.env.overwriteOutput = True


def calc_mk_trend(input_md, output_path, boundary, mask_by_sig):
    # --- Name the trend output CRF ---
    base_name = os.path.splitext(os.path.basename(input_md))[0]
    trend_crf = os.path.join(output_path, f"{base_name}_mk_trend.crf")

    msg("Generating Mann-Kendall trend...")
    trend_raster = arcpy.ia.GenerateTrend(
        raster=input_md,
        dimension_name="StdTime",
        regression_type="MANN-KENDALL",
        rmse=True,
        r2=True,
        slope_p_value=True
    )

    trend_raster.save(trend_crf)
    msg(f"Mann-Kendall trend raster saved to: {trend_crf}")

    if not mask_by_sig:
        msg("Skipping statistical significance masking.")
        return

    msg("Extracting statistically significant Sen's slope (p < 0.05)...")

    # Path to the P-Value band
    pval_band = os.path.join(trend_crf, "P_Value")

    if not arcpy.Exists(pval_band):
        raise RuntimeError(f"P_Value band not found: {pval_band}")

    # Create mask from p-value threshold
    sig_mask = arcpy.sa.ExtractByAttributes(pval_band, "VALUE < 0.05")

    # Apply mask to the full CRF
    sig_pixels = arcpy.sa.ExtractByMask(trend_crf, sig_mask, analysis_extent=boundary)

    # Extract Sen’s slope (Band 1)
    sens_slope = arcpy.sa.ExtractBand(sig_pixels, 1)

    # Output path
    sig_output = os.path.join(output_path, f"{base_name}_mk_trend_sig.crf")
    sens_slope.save(sig_output)

    msg(f"Significant Sen's slope saved to: {sig_output}")

    # Does sig_pixels contain ANY non-null values?
    sig_na_result = arcpy.GetRasterProperties_management(sig_output, "ALLNODATA")
    sig_na = sig_na_result.getOutput(0)
    if sig_na == '1':
        warn(f'"{sig_pixels}" has no significant pixels. Hotspots of significant change cannot be generated in tool 4) Map Hotspots of Significant Change.')
        
    # Clean up temp rasters
    arcpy.management.Delete([sig_mask, sig_pixels])



