"""
DESERTIFICATION PROCESS INVESTIGATION ON LORDSBURG PLAYA
Toolbox developed by Jack Taylor (Programmer - Earth Data Analysis Center (EDAC)) for the Southern Plains Transportation Center (SPTC)
as a component of Task 4. 

November 2025

This toolbox replicates the analysis performed for the project and is designed to be used within ArcGIS Pro.
Data outputs from the original analysis are available at: *************

"""

import arcpy
import os
from utils import msg # GUI messaging utility
from MakeMultidimWithLocalFiles import make_multidim_from_local_files # source for 1 - Make Multidimensional Raster from Local Files
from CalculateIndex import calc_md_index # source for 2 - Calculate Spectral Index
from CalculateMkTrend import calc_mk_trend # source for 3 - Calculate Mann-Kendall Trend
from MapHotspots import map_hotspots # source for 4 - Map hotspots of significant change


class Toolbox:
    def __init__(self):
        """Define the toolbox (the name of the toolbox is the name of the
        .pyt file)."""
        self.label = "Lordsburg Playa Trend Analysis Tools"
        self.alias = "LordsburgPlayaTrendAnalysisTools"

        # List of tool classes associated with this toolbox
        self.tools = [makeMultidim, calcIndex, calcMKTrend, mapHotspots]



class makeMultidim:
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "1) Make Multidimensional Raster from Local Files"
        self.description = "Use a set of local cloud-free Landsat Surface Reflectance band files (ex: extracted .tar packages from https://earthexplorer.usgs.gov/) to create a multidimensional raster in a geodatabase."

    def getParameterInfo(self):
        """Define the tool parameters."""

        # Output geodatabase
        wrkSpc = arcpy.Parameter(
            name='Output_GDB',
            displayName='Output GDB',
            direction='Input',
            datatype='DEWorkspace',
            parameterType='Required'
            )
        
        # Input data folder
        data_path = arcpy.Parameter(
            name='Input_Folder',
            displayName='Input Folder',
            direction='Input',
            datatype='DEFolder',
            parameterType='Required'
            )
        # Output mosaic name
        mosaic_name = arcpy.Parameter(
            name='Mosaic_Name',
            displayName='Mosaic Name',
            direction='Input',
            datatype='GPString',
            parameterType='Required'
            )
        
        # Variable name
        variable_name = arcpy.Parameter(
            name='Variable_Name',
            displayName='Variable Name',
            direction='Input',
            datatype='GPString',
            parameterType='Required'
            )
        
        # Boundary shapefile or similar
        boundary = arcpy.Parameter(
            name='Boundary',
            displayName='Boundary',
            direction='Input',
            datatype='GPFeatureLayer',
            parameterType='Required'
            )

        params = [wrkSpc, data_path, mosaic_name, variable_name, boundary]
        return params

    def isLicensed(self):
        """Set whether the tool is licensed to execute."""
        return True

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter. This method is called after internal validation."""
        return

    def execute(self, parameters, messages):
        """The source code of the tool."""

        wrkSpc = parameters[0].valueAsText
        data_path = parameters[1].valueAsText
        mosaic_name = parameters[2].valueAsText
        variable_name = parameters[3].valueAsText
        boundary = parameters[4].valueAsText

        make_multidim_from_local_files(wrkSpc, data_path, mosaic_name, variable_name, boundary)
        return

    def postExecute(self, parameters):
        """This method takes place after outputs are processed and
        added to the display."""
        return



class calcIndex:
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "2) Calculate Spectral Index"
        self.description = "Use the mosaic dataset created in Step 1 to calculate normalized difference indices (NDVI, NDMI, NDDSI) and export as multidimensional rasters in cloud-reference-format (CRF)."

    def getParameterInfo(self):
        """Define the tool parameters."""
        mosaic_param = arcpy.Parameter(
            name="Input_Mosaic",
            displayName="Input Mosaic Dataset (Generated in Step 1)",
            direction="Input",
            datatype="GPRasterLayer",
            parameterType="Required"
            )

        ndvi_param = arcpy.Parameter(
            name="Calc_NDVI",
            displayName="Calculate NDVI",
            direction="Input",
            datatype="GPBoolean",
            parameterType="Optional"
            )
        ndvi_param.value = True

        ndmi_param = arcpy.Parameter(
            name="Calc_NDMI",
            displayName="Calculate NDMI",
            direction="Input",
            datatype="GPBoolean",
            parameterType="Optional"
            )
        ndmi_param.value = True 

        nddsi_param = arcpy.Parameter(
            name="Calc_NDDSI",
            displayName="Calculate NDDSI",
            direction="Input",
            datatype="GPBoolean",
            parameterType="Optional"
            )
        nddsi_param.value = True

        params = [mosaic_param, ndvi_param, ndmi_param, nddsi_param]
        return params

    def isLicensed(self):
        """Set whether the tool is licensed to execute."""
        ext_ok = arcpy.CheckExtension("Spatial") == "Available"
        return ext_ok

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter. This method is called after internal validation."""
        if arcpy.CheckExtension("Spatial") != "Available":
                parameters[0].setErrorMessage(
                "Spatial Analyst extension is required.\n\n"
                "Please enable the Spatial Analyst extension (Project > Licensing) "
                "before running this tool."
            )
        return

    def execute(self, parameters, messages):
        """The source code of the tool."""
        mosaic_path = parameters[0].valueAsText
        ndvi_toggle = parameters[1]
        ndmi_toggle = parameters[2]
        nddsi_toggle = parameters[3]

        output_dir = arcpy.mp.ArcGISProject("CURRENT").homeFolder + r"\output_indices"
        if not os.path.exists(output_dir):
            os.makedirs(output_dir)

        msg(f"Output directory: {output_dir}")
        msg("Each index may take several minutes to process depending on the size of the input mosaic dataset.")

        calc_md_index(mosaic_path, output_dir, ndvi_toggle, ndmi_toggle, nddsi_toggle)
        return

    def postExecute(self, parameters):
        """This method takes place after outputs are processed and
        added to the display."""
        return
    

class calcMKTrend:
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "3) Calculate Mann-Kendall Trend"
        self.description = "The Mann-Kendall trend test is a non-parametric test used to identify trends in time series data. This tool calculates the Mann-Kendall trend for a multidimensional raster and extracts statistically significant Sen's slope values."

    def getParameterInfo(self):
        """Define the tool parameters."""
        input_md = arcpy.Parameter(
            name="Input_Mosaic",
            displayName="Input Multidimensional Index Dataset (Generated in Step 2)",
            direction="Input",
            datatype="GPRasterLayer",
            parameterType="Required"
            )

        # Boundary shapefile
        boundary = arcpy.Parameter(
            name='Boundary',
            displayName='Area of Interest Boundary',
            direction='Input',
            datatype='GPFeatureLayer',
            parameterType='Required'
            )

        params = [input_md, boundary]
        return params

    def isLicensed(self):
        """Set whether the tool is licensed to execute."""
        ext_ok = arcpy.CheckExtension("ImageAnalyst") == "Available"
        return ext_ok

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        if arcpy.CheckExtension("ImageAnalyst") != "Available":
            parameters[0].setErrorMessage(
            "Image Analyst extension is required.\n\n"
            "Please enable the Image Analyst extension (Project > Licensing) "
            "before running this tool."
            )
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter. This method is called after internal validation."""
        return

    def execute(self, parameters, messages):
        """The source code of the tool."""
        input_md = parameters[0].valueAsText
        boundary = parameters[1].valueAsText  

        output_dir = arcpy.mp.ArcGISProject("CURRENT").homeFolder + r"\output_trends"
        if not os.path.exists(output_dir):
            os.makedirs(output_dir)  

        msg(f"Output directory: {output_dir}")
        calc_mk_trend(input_md, output_dir, boundary, mask_by_sig=True)  

        return

    def postExecute(self, parameters):
        """This method takes place after outputs are processed and
        added to the display."""
        return
    
class mapHotspots:
    def __init__(self):
        """Define the tool (tool name is the name of the class)."""
        self.label = "4) Map hotspots of significant change"
        self.description = "This tool uses outputs from the Mann-Kendall test to detirmine what locations have had significant clustering of positive and negative change."

    def getParameterInfo(self):
        """Define the tool parameters."""

        # Output geodatabase
        wrkSpc = arcpy.Parameter(
            name='Output_GDB',
            displayName='Output GDB',
            direction='Input',
            datatype='DEWorkspace',
            parameterType='Required'
            )
        
        # Boundary shapefile
        boundary = arcpy.Parameter(
            name='Boundary',
            displayName='Area of Interest Boundary',
            direction='Input',
            datatype='GPFeatureLayer',
            parameterType='Required'
            )
        
        # Raster of significant pixels for a given index
        sig_pixels = arcpy.Parameter(
            name='Sig_Pixels',
            displayName='Significant Pixels Raster (Generated in Step 3)',
            direction='Input',
            datatype='GPRasterLayer',
            parameterType='Required'
            )

        # H3 Polygon resolution for hotspot tesselation - default value of 7
        h3_res = arcpy.Parameter(
            name='H3_Res',
            displayName='H3 Hexagon Resolution (0 - 15: scale for hotspots)',
            direction='Input',
            datatype='GPLong',
            parameterType='Required'
            )
        h3_res.value = 7
        # Restrict allowed values to 0–15

        # Output name
        output_name = arcpy.Parameter(
            name='output_fc_name',
            displayName='Output Feature Class Name',
            direction='Input',
            datatype='GPString',
            parameterType='Required'
            )


        params = [wrkSpc, boundary, sig_pixels, h3_res, output_name]
        return params

    def isLicensed(self):
        """Set whether the tool is licensed to execute."""
        ext_ok = arcpy.CheckExtension("Spatial") == "Available"
        return ext_ok

    def updateParameters(self, parameters):
        """Modify the values and properties of parameters before internal
        validation is performed.  This method is called whenever a parameter
        has been changed."""
        if arcpy.CheckExtension("Spatial") != "Available":
            parameters[0].setErrorMessage(
            "Spatial Analyst extension is required.\n\n"
            "Please enable the Spatial Analyst extension (Project > Licensing) "
            "before running this tool."
        )      
        return

    def updateMessages(self, parameters):
        """Modify the messages created by internal validation for each tool
        parameter. This method is called after internal validation."""
        return

    def execute(self, parameters, messages):
        """The source code of the tool."""
        wrkspc = parameters[0].valueAsText
        boundary = parameters[1].valueAsText
        sig_pixels = parameters[2].valueAsText
        h3_res = int(parameters[3].valueAsText)
        output_name = parameters[4].valueAsText

        map_hotspots(wrkspc, boundary, sig_pixels, h3_res, output_name)

        return

    def postExecute(self, parameters):
        """This method takes place after outputs are processed and
        added to the display."""
        return