import arcpy
import os
import datetime
from utils import msg, warn
arcpy.env.overwriteOutput = True

def make_multidim_from_local_files(wrkspc, data_path, mosaic_name, variable_name, boundary): 
    # Use study area to create a crs for output dataset
    boundary_desc = arcpy.Describe(boundary)
    boundary = boundary_desc.catalogPath
    crs = arcpy.Describe(boundary).spatialReference

    # Dictionary to hold lists of band paths per scene
    scene_bands = {}

    # Iterate through subfolders
    for folder_name in os.listdir(data_path):
        scene_path = os.path.join(data_path, folder_name)
        
        # Skip if it's not a directory
        if not os.path.isdir(scene_path):
            continue

        # Find all Surface Reflectance band files
        sr_bands = [
            os.path.join(scene_path, f)
            for f in os.listdir(scene_path)
            if f.endswith(".TIF") and "_SR_B" in f
        ]

        # Sort by band number
        sr_bands.sort(key=lambda x: int(os.path.basename(x).split("_B")[1].split(".")[0]))

        # Store in dictionary
        scene_bands[folder_name] = sr_bands

    # Formatting output
    for scene, bands in scene_bands.items():
        msg(f"\nScene: {scene}")
        for b in bands:
            msg(f"  {b}")

    # Composite bands in geodatabase
    for scene in scene_bands:
        msg(f'Importing {scene} into ArcGIS...')
        out_name = os.path.join(wrkspc, str(scene))
        if not arcpy.Exists(out_name):
            arcpy.management.CompositeBands(scene_bands[scene], out_name)

    # Create mosaic dataset
    msg(f'Creating mosaic dataset in "{wrkspc}"')
    arcpy.management.CreateMosaicDataset(wrkspc, mosaic_name, crs, num_bands=7)
    for scene in scene_bands:
        path = os.path.join(wrkspc, scene)
        arcpy.management.AddRastersToMosaicDataset(os.path.join(wrkspc, mosaic_name), "Raster Dataset", path)

    md_path = os.path.join(wrkspc, mosaic_name)
    # Build multidim info
    if "date" not in [f.name for f in arcpy.ListFields(md_path)]:
        arcpy.management.AddField(md_path, "date", "DATE")

    if "variable" not in [f.name for f in arcpy.ListFields(md_path)]:
        arcpy.management.AddField(md_path, "variable", 'TEXT')

    fields = ["Name", "date", "variable"]
    with arcpy.da.UpdateCursor(md_path, fields) as cur:
        for row in cur:
            row[2] = variable_name
            datetxt = row[0].split("_")[3]
            row[1] = datetime.datetime.strptime(f"{datetxt}", "%Y%m%d")
            cur.updateRow(row)


    arcpy.management.BuildMultidimensionalInfo(md_path, variable_field="variable", dimension_fields="date")
    msg(f'Output multidimensional mosaic dataset created in "{os.path.join(wrkspc, mosaic_name)}"')

