import arcpy
import arcpy.sa
import sys
from utils import msg, warn

arcpy.env.overwriteOutput = True

class NoSig(Exception):
    pass


def map_hotspots(wrkspc, boundary, sig_pixels, h3_res, output_fc_name):
    """
    Using the Sen's Slope value of identified pixels of significant change, 
    convert these pixels to points, aggregate points to H3 hexagons, 
    and calculate both positive and negative hotspots.
    Outputs two feature classes, one for positive change and one for negative change. 
    """
    arcpy.env.workspace = wrkspc
    crs = arcpy.Describe(boundary).spatialReference

    try:
        # Does sig_pixels contain ANY non-null values?
        sig_na_result = arcpy.GetRasterProperties_management(sig_pixels, "ALLNODATA")
        sig_na = sig_na_result.getOutput(0)
        if sig_na == '1':
            arcpy.AddError(f'"{sig_pixels}" has no significant pixels. Hotspots of significant change cannot be generated.')
            raise NoSig(Exception)
    except NoSig:
        sys.exit()
    
    

    # Extract positive and negative trends into seperate rasters
    pos_q = "VALUE >= 0"
    neg_q = "VALUE < 0"

    msg("Separating positive and negative trends...")
    pos_pix = arcpy.sa.ExtractByAttributes(sig_pixels, pos_q)
    neg_pix = arcpy.sa.ExtractByAttributes(sig_pixels, neg_q)
    arcpy.management.CalculateStatistics(pos_pix, skip_existing="OVERWRITE")
    arcpy.management.CalculateStatistics(neg_pix, skip_existing="OVERWRITE")

    # Convert pixels to points
    arcpy.conversion.RasterToPoint(pos_pix, "pos_points", "VALUE")
    arcpy.conversion.RasterToPoint(neg_pix, "neg_points", "VALUE")

    # Create h3 hexagon tesselation for boundary, allow user to set level
    msg("Creating hexagon tesselation...")
    arcpy.management.GenerateTessellation("h3_polygons", boundary, "H3_HEXAGON", Spatial_Reference=crs, H3_Resolution=h3_res) # type: ignore
    msg("Hexagons generated.")

    # Join points to hexagons
    msg("Calculating hotspots...")
    arcpy.analysis.SpatialJoin(target_features="h3_polygons",
                            join_features="pos_points",
                            out_feature_class="pos_count_per_hex",
                            join_operation="JOIN_ONE_TO_ONE",
                            join_type="KEEP_ALL",
                            match_option="INTERSECT")

    arcpy.analysis.SpatialJoin(target_features="h3_polygons",
                            join_features="neg_points",
                            out_feature_class="neg_count_per_hex",
                            join_operation="JOIN_ONE_TO_ONE",
                            join_type="KEEP_ALL",
                            match_option="INTERSECT")

    # Calculate hotspots using join count per hexagon
    arcpy.stats.HotSpots(Input_Feature_Class="pos_count_per_hex",
                        Input_Field="JOIN_COUNT",
                        Output_Feature_Class=output_fc_name + "_positive",
                        Conceptualization_of_Spatial_Relationships="INVERSE_DISTANCE",
                        Distance_Method="EUCLIDEAN_DISTANCE")
    msg(f'Positive hotspots exported to: {arcpy.Describe(output_fc_name + "_positive").catalogPath}')

    arcpy.stats.HotSpots(Input_Feature_Class="neg_count_per_hex",
                        Input_Field="JOIN_COUNT",
                        Output_Feature_Class=output_fc_name + "_negative",
                        Conceptualization_of_Spatial_Relationships="INVERSE_DISTANCE",
                        Distance_Method="EUCLIDEAN_DISTANCE")
    msg(f'Negative hotspots exported to: {arcpy.Describe(output_fc_name + "_negative").catalogPath}')

    msg("Cleaning up...")
    arcpy.management.Delete([pos_pix, neg_pix, "pos_points", "neg_points"])
    msg('Done!')
    return
